/* IDE Load - IDE controler load monitor 
 * Copyright (C) 2001  Emmanuel VARAGNAT <coredump@free.fr>
 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdlib.h>
#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>
#include <signal.h>
#include <string.h>
#include <math.h>
#include <errno.h>

#include <X11/Xlib.h>
#include <X11/X.h>

#include <list>
#include <set>

#include "circ_buff.h"
#include "ide_load.h"
#include "IrqAct.h"
#include "IOAct1.h"

/*****************************
 * DEFINITIONS
 *****************************/

#define INTERRUPT        14
#define DEFAULT_TIMEOUT  2
#define DEFAULT_WIDTH    100
#define DEFAULT_HEIGHT   100

#define BG_COLOR         "grey80"
#define FG_COLOR         "black"
#define GR_COLOR1        "PaleGreen"
#define GR_COLOR2        "MediumSeaGreen"

/****************************
 * GLOBALS
 ****************************/

Display           *dpy;
Window            mainwin;
GC                gc, gcwhite, gctext, gcfg1, gcfg2;
unsigned int      timeout;
Font              font_id;
XFontStruct       *fst;
int               signal_happen;
XColor            white, black, fgcol1, fgcol2, bgcol;

list<PixBuff*>            view_list;
list<PixBuff*>::iterator  it;
PixBuff                   *active;

set<string>       item_set;
int               nb_items;

char              *appname;
char              *bg_color, *fg_color;
char              *gr1_color, *gr2_color;

/****************************
 * FUNCTIONS
 ****************************/

static void usage() {
  cout << endl
       << "This shows IRQ activity of IDE controlers and global read/write" << endl
       << "activity (SCSI, IDE, ...)." << endl
       << "You can switch between different views with the right button." << endl
       << endl
       << "Usage : " << appname << " [options] [ io | ide0 | ide1 | ... ]" << endl
       << "  -update : set the update period (seconds)" << endl
       << "  -bg     : set background color" << endl
       << "  -fg     : set foreground color" << endl
       << "  -gr1    : set read and interrupt load color" << endl
       << "  -gr2    : set write load color" << endl
       << "  -help   : don't know" << endl
       << endl;
}

/**********************************
 **********************************
 **********************************/

void find_ide_irq() {
  char          buffer[255];
  unsigned int  i, j;
  char          ch;
  int           fd;

  fd = open("/proc/interrupts", O_RDONLY);
  if(fd < 0) {
    perror("open");
    exit(0);
  }

  i = 0;
  while(read(fd, &ch, 1) > 0) {
    switch(ch) {
    case '\n':
      buffer[i] = '\0';
      for(j = i-1; buffer[j] >= '0' && buffer[j] <= '9'; j--);

      if(j != i-1 && strncmp(buffer+j-2, "ide", 3) == 0) {
	unsigned int irq;
	set<string>::iterator elem;

	for(i = 0; buffer[i] != ':'; i++);
	buffer[i] = '\0';
	sscanf(buffer, "%u", &irq);

	elem = item_set.find(buffer+j-2);
	if(nb_items == 0 || (nb_items && elem != item_set.end())) {
	   view_list.push_back(new IrqAct(dpy, mainwin,
					  DEFAULT_WIDTH, DEFAULT_HEIGHT,
					  irq, strdup(buffer+j-2),
					  100,
					  fg_color, bg_color, gr1_color));
	   if(elem != item_set.end())
	     item_set.erase(elem);
	}
      }
      i = 0;
      break;
    default:
      buffer[i++] = ch;
      break;
    }
  }
}

static void parse_cmdline(int argc, char **argv) {
  int i;

  for(i = 1; i < argc; i++) {
    if(strcmp(argv[i], "-help") == 0) {
	goto usage;
    }
    else if(strcmp(argv[i], "-bg") == 0) {
      if(++i < argc)
	bg_color = argv[i];
      else
	goto usage;
    }
    else if(strcmp(argv[i], "-fg") == 0) {
      if(++i < argc)
	fg_color = argv[i];
      else
	goto usage;
    }
    else if(strcmp(argv[i], "-gr1") == 0) {
      if(++i < argc)
	gr1_color = argv[i];
      else
	goto usage;
    }
    else if(strcmp(argv[i], "-gr2") == 0) {
      if(++i < argc)
	gr2_color = argv[i];
      else
	goto usage;
    }
    else if(strcmp(argv[i], "-update") == 0) {
      unsigned int tmp;

      if(++i >= argc)
	goto usage;

      if(sscanf(argv[i], "%u", &tmp))
	timeout = tmp;
      else {
	cout << "Bad update value" << endl;
      }
    }
    else
      item_set.insert(argv[i]);
  }
  
  nb_items = item_set.size();
  return;

 usage:
  usage();
  exit(1);
}

static void ShowXError(const char *txt, Status status) {
  char str[255];

  str[0] = '\0';
  XGetErrorText(dpy, status, str, 254);
}



/* The alarm signal interrupt the select system call */
static void signal_event(int n) {
  signal_happen = 1;
  alarm(timeout);
}


/*
 * Based on DA_NextEventOrTimeout
 * written by Alfredo K. Kojima
 */
static int NextEventOrTimeout (XEvent * event, unsigned long miliseconds) {
  static struct timeval timeout;
  fd_set rset;

  if(signal_happen)
    return 2;

  XSync (dpy, False);
  if (XPending (dpy)) {
    XNextEvent (dpy, event);
    return 0;
  }

  timeout.tv_sec = miliseconds / 1000;
  timeout.tv_usec = (miliseconds % 1000) * 1000;

  FD_ZERO (&rset);
  FD_SET (ConnectionNumber (dpy), &rset);

  errno = 0;

  switch(select (ConnectionNumber (dpy) + 1, &rset, NULL, NULL, &timeout)) {
  case 0:
    return 1;

  case -1:
    if(errno == EINTR) /* interrupted by the SIGALRM */
      return 2;
    else
      return -1;
    cout << "ERROR(select) " << strerror(errno) << endl;
    return False;

  default:
    XNextEvent (dpy, event);
    return 0;
  }
}

static void event_received(XEvent *ev) {
  switch(ev->type) {
  case Expose:
    XCopyArea(dpy,
	      active->pix, mainwin, 
	      DefaultGC(dpy, DefaultScreen(dpy)),
	      0, 0,
	      active->width, active->height,
	      0, 0);
    break;

  case ConfigureNotify:
    {
      unsigned int w, h;

      w = ((XConfigureEvent*)ev)->width;
      h = ((XConfigureEvent*)ev)->height;
      if(active->width != w || active->height != h) {
	for(list<PixBuff*>::iterator i=view_list.begin(); i != view_list.end(); i++)
	  (*i)->resize(w, h);

	ShowXError("XResizeWindow", XResizeWindow(dpy, mainwin, w, h));
	
	active->draw();
	
	XCopyArea(dpy,
		  active->pix, mainwin, 
		  DefaultGC(dpy, DefaultScreen(dpy)),
		  0, 0,
		  w, h,
		  0, 0);
      }
    }
    break;

  case ButtonPress:
    if(((XButtonPressedEvent*)ev)->button == Button3) {
      it++;
      
      if(it == view_list.end())
	it = view_list.begin();

      active = *it;
      
      active->draw();
      XCopyArea(dpy,
		active->pix, mainwin, 
		DefaultGC(dpy, DefaultScreen(dpy)),
		0, 0,
		active->width, active->height,
		0, 0);
    }
    break;
  default:
    break;
  }
}

/*****************************
 *  MAIN
 *****************************/

int main(int argc, char *argv[]) {
  XEvent      ev;
  XColor unused, bgcol;

  appname = argv[0];

  timeout = DEFAULT_TIMEOUT;

  bg_color = BG_COLOR;
  fg_color = FG_COLOR;
  gr1_color = GR_COLOR1;
  gr2_color = GR_COLOR2;

  parse_cmdline(argc, argv);

  dpy = XOpenDisplay(NULL);
  if(dpy == NULL)
    exit(1);

  if(XAllocNamedColor(dpy, DefaultColormap(dpy, DefaultScreen(dpy)),
		      bg_color, &bgcol, &unused) == 0)
    {
      cout << " Bad background color name" << endl;
      XCloseDisplay(dpy);
      exit(1);
    }

  mainwin = XCreateSimpleWindow(dpy, DefaultRootWindow(dpy),
				0, 0,
				DEFAULT_WIDTH, DEFAULT_HEIGHT,
				0, 0,
				bgcol.pixel);
  XSelectInput(dpy, mainwin, ExposureMask | ButtonPressMask);
  XStoreName(dpy, mainwin, "IDEload");

  try { 
    find_ide_irq();
    if(nb_items == 0 || (nb_items && item_set.find("io") != item_set.end())) {
	view_list.push_front(new IOAct1(dpy, mainwin,
				       DEFAULT_WIDTH, DEFAULT_HEIGHT,
				       100,
				       fg_color, bg_color,
				       gr1_color, gr2_color));
    }
  }
  catch(BadColorName &e) {
    cout << " Bad " << e.msg << " color name " << endl;
    XCloseDisplay(dpy);
    exit(1);
  }

  it = view_list.begin();
  active = *it;

  ShowXError("XMapWindow", XMapWindow(dpy, mainwin));

  XFlush(dpy);

  signal(SIGALRM, signal_event);
  alarm(timeout);
  signal_happen = 0;

  for(;;) {
    switch(NextEventOrTimeout(&ev, timeout*1000)) {
    case -1: /* ERROR */
      break;
    case 0:
      event_received(&ev);
      break;
    case 2:
      signal_happen = 0;
    case 1:
      for(list<PixBuff*>::iterator i=view_list.begin(); i != view_list.end(); i++)
	(*i)->update();

      XCopyArea(dpy,
		active->pix, mainwin, 
		DefaultGC(dpy, DefaultScreen(dpy)),
		0, 0,
		active->width, active->height,
		0, 0);
      break;
    }
  }

  XCloseDisplay(dpy);

  return 0;
}
