/* Circular buffer 
 * Copyright (C) 2001  Emmanuel VARAGNAT <coredump@free.fr>
 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include "circ_buff.h"

#define BUFF_INCR        100
#define MIN(a,b) ((a<b) ? a : b)

int cbuff_init(struct circ_buff *buff, unsigned int win) {
  unsigned int blen;

  blen = ((win/BUFF_INCR) + 1) * BUFF_INCR;
  buff->buff = (unsigned int*) malloc(blen*sizeof(unsigned int));

  if(buff->buff == NULL)
    return 1;

  buff->len = blen;
  buff->window = win;
  buff->begin = buff->end = buff->buff;
  buff->full = 0;

  return 0;
}

int cbuff_resize(struct circ_buff *buff, unsigned int new_win) {
  unsigned int size1, size2, size3;
  unsigned int size, data_size;
  unsigned int *p;

  if(new_win < buff->window) {
    unsigned int diff, cpy;

    data_size = cbuff_real_size(buff);

    diff = buff->window - new_win;

    if(buff->begin >= buff->end) {
      cpy = MIN((buff->buff + new_win) - buff->end,
		(buff->buff + buff->window) - buff->begin);
      buff->begin = (buff->buff + new_win) - cpy;
      memcpy(buff->begin,
	     (buff->buff + buff->window) - cpy,
	     cpy*sizeof(unsigned int));
      buff->full  = (buff->begin == buff->end);
    }
    else {
      cpy = MIN(new_win, cbuff_real_size(buff));
      memcpy(buff->buff, buff->end-cpy, cpy*sizeof(unsigned int));
      buff->begin = buff->buff;
      buff->end   = (cpy == new_win) ? buff->begin : buff->buff + cpy;
      buff->full  = (cpy == new_win);
    }

    /* no realloc if the new size is greater than the third of old size
       and if have to do too much memory copy */
    if(new_win < buff->len/3) {
      size = ((new_win / BUFF_INCR) + 1) * BUFF_INCR;
      if(buff->begin != buff->buff) {
	/*memcpy();*/
      }

      /* Realloc */
      p = (unsigned int*) realloc(buff->buff, size*sizeof(unsigned int));
      /* we don't test p because realloc shouldn't fail to get less space */
    }

    buff->window = new_win;
  }

  if(new_win > buff->window) {
    if(new_win > buff->len) {
      size = ((new_win / BUFF_INCR) + 1) * BUFF_INCR;
      p = (unsigned int*) realloc(buff->buff, size*sizeof(unsigned int));
      if(p == NULL)
	return -1;

      buff->begin = p + (buff->begin - buff->buff);
      buff->end   = p + (buff->end - buff->buff);
      buff->buff  = p;
    }

    if( buff->begin != buff->buff && (buff->full || buff->end < buff->begin) ) {
      size1 = buff->end - buff->buff;
      size2 = (new_win - buff->window);

      if(size1 < size2) {
	memcpy(buff->buff + buff->window, buff->buff, size1*sizeof(unsigned int));
	buff->end = buff->begin + cbuff_real_size(buff);
      }
      else {
	memcpy(buff->buff + buff->window, buff->buff, size2*sizeof(unsigned int));
	size3 = buff->end - (buff->buff + size2);
	memmove(buff->buff, buff->buff + size2, size3*sizeof(unsigned int));
	buff->end = buff->buff + size3;
      }
    }
      
    buff->window = new_win;
    buff->full = 0;
  }

  return 0;
}
