/*
 * IRQ activity view for one IDE controler.
 *
 * Copyright(C) 2001 Emmanuel VARAGNAT <coredump@free.fr>
 * licensed under the GPL
 */

#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include "IrqAct.h"

#include <iostream>

enum int_state { ST_BEGIN, ST_END, ST_INT, ST_VAL };

IrqAct::IrqAct(Display *dpy, Window win,
	       unsigned int w, unsigned int h,
	       unsigned int interrupt, char *ide_name,
	       unsigned int size = 100,
	       char *fg_str = NULL, char *bg_str = NULL,
	       char *graph = NULL)
  : PixBuff(dpy, win, w, h, fg_str, bg_str)
{
  XGCValues gcv;
  Font font_id;
  XColor color, unused;

  irq = interrupt;
  name = ide_name;
  cbuff_init(&buff, size);
  get_val(&last);
  max = 0;

  font_id = XLoadFont(dpy, "-*-helvetica-medium-r-*-*-8-*-*-*-*-*-iso8859-*");
  fst = XQueryFont(dpy, font_id);

  gcv.font = font_id;
  gcv.line_width = 1;
  XChangeGC(dpy, gc_fg, GCFont | GCLineWidth, &gcv);

  if(graph == NULL)
    gc_graph = gc_fg;
  else {
    if(XAllocNamedColor(dpy, DefaultColormap(dpy, DefaultScreen(dpy)),
			graph, &color, &unused) == 0)
      {
	throw BadColorName("graphic1");
      }
    gcv.foreground = color.pixel;
    gc_graph = XCreateGC(dpy, win,
			 GCForeground,
			 &gcv);
  }
}


int IrqAct::get_val(unsigned long long *val) {
  int fd;
  char str_int[10], buff[255];
  char ch;
  unsigned int i;
  enum int_state state;

  sprintf(str_int, "%u", irq);

  fd = open("/proc/interrupts", O_RDONLY);
  if(fd < 0) {
    perror("open");
    exit(0);
  }

  i = 0;  /* for compiler warning */
  state = ST_BEGIN;
  *val = 0;

  while(read(fd, &ch, 1) > 0) {
    switch(state) {
    case ST_BEGIN:
      switch(ch) {
      case ' ':
      case '\t':
      case '\n':
	continue;
      default:
        if(ch >= '0' && ch <='9') {
	  state = ST_INT;
	  buff[0] = ch;
	  i = 1;
	}
	else
	  state = ST_END;
      }
      break;
    case ST_INT:
      if(ch >= '0' && ch <= '9') {
	buff[i++] = ch;
	break;
      }
      if(ch == ':') {
        buff[i] = '\0';
	if(strcmp(buff, str_int) == 0)
	  state = ST_VAL;
	else
	  state = ST_END;
      }
      else
	state = ST_END;
      break;
    case ST_VAL:
      if(ch == '\n')
	goto error;

      if(*val && (ch < '0' || ch > '9') )
	goto end;

      if(ch >= '0' && ch <= '9')
	*val = (*val)*10 + (ch - '0');
      break;
    case ST_END:
      if(ch == '\n')
	state = ST_BEGIN;
      break;
    }
  }

  return 2;
 
 error:
  close(fd);
  return 1;

 end:
  close(fd);
  return 0;
}

void IrqAct::draw() {
  char           strval[20];
  unsigned int   i, n, len;
  unsigned int   size, offset;
  unsigned int   half_asc, half;
  unsigned int   top, bottom;
  unsigned int   start, stop;
  int            asc, desc, dir;
  XCharStruct    cst;

  /* efface le fond */
  XFillRectangle(dpy, pix, gc_bg, 0, 0, width, height);

  /* dessine l'chelle sur le cot et la borne maximale */
  n = sprintf(strval, "%llu", max);
  XTextExtents(fst, " ", 3, &dir, &asc, &desc, &cst);
  XDrawString(dpy, pix, gc_fg,
	      2, 2 + asc,
	      strval, n);
  offset = 2 + XTextWidth(fst, strval, n) + 2;

  half_asc = asc >> 1;

  /* Draw ticks and axis */
  top = 2 + half_asc;
  bottom = height - 1;
  XDrawLine(dpy, pix, gc_fg, offset, top,    offset + 3, top);
  XDrawLine(dpy, pix, gc_fg, offset, bottom, offset + 3, bottom);
  half = top + (bottom - top) / 2;
  XDrawLine(dpy, pix, gc_fg, offset, half, offset + 3, half);

  XDrawLine(dpy, pix, gc_fg, offset + 3, top, offset + 3, bottom + 1);
  offset += 5;

  size = cbuff_real_size(&buff);

  /* Draw graph */
  n = width - offset;
  start = (size < n) ? 0 : size-n;
  stop = size - start;
  for(i = 0; i < stop; i++) {
    /* WARNING : a division by zero may occur */
    XDrawLine(dpy, pix, gc_graph,
	      i + offset, bottom,
	      i + offset, bottom - (int)((double)cbuff_get(&buff, start+i)/(double)max*(double)(bottom - top)));
  }

  /* Draw IDE name */
  len = strlen(name);
  XDrawString(dpy, pix, gc_fg,
	      width - XTextWidth(fst, name, len) - 1,
	      2 + asc,
	      name, len);
}

void IrqAct::update() {
  unsigned long long  count;
  unsigned long       diff;
  unsigned int        old_last;

  count = 0;

  get_val(&count);
  diff = count-last;

  old_last = cbuff_get(&buff, 0);
  cbuff_add(&buff, diff);

  if(cbuff_real_size(&buff) == buff.window && old_last == max/* && diff < max*/) {
    unsigned int i, size, val;

    size = cbuff_real_size(&buff);
    max = 0;
    for(i = 0; i < size; i++) {
      val = cbuff_get(&buff, i);
      max = (val > max) ? val : max;
    }
  }
  else
    max = (max < diff) ? diff : max;

  last = count;

  draw();
}

void IrqAct::resize(unsigned int w, unsigned int h) {
  PixBuff::resize(w, h);
  cbuff_resize(&buff, w);
}

