/*
 * General R/W IO view, with read and write on the same histogram.
 *
 * Copyright(C) 2001 Emmanuel VARAGNAT <coredump@free.fr>
 * licensed under the GPL
 */

#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include "IOAct1.h"

#include <iostream>

IOAct1::IOAct1(Display *dpy, Window win,
	       unsigned int w, unsigned int h,
	       unsigned int size = 100,
	       char *fg_str = NULL, char *bg_str = NULL,
	       char *rd_str = NULL, char *wr_str = NULL)
  : PixBuff(dpy, win, w, h, fg_str, bg_str)
{
  XGCValues gcv;
  Font font_id;
  unsigned long rio_val, wio_val;
  char *default_rd = "#888888";
  char *default_wr = "black";
  XColor color,unused;

  cbuff_init(&io, size);
  cbuff_init(&rio, size);
  get_val(&rio_val, &wio_val);
  lastread = rio_val;
  lasttot = rio_val + wio_val;
  max = 0;

  font_id = XLoadFont(dpy, "-*-helvetica-medium-r-*-*-8-*-*-*-*-*-iso8859-*");
  fst = XQueryFont(dpy, font_id);

  gcv.font = font_id;
  gcv.line_width = 1;
  XChangeGC(dpy, gc_fg, GCFont | GCLineWidth, &gcv);

  if(rd_str == NULL)
    rd_str = default_rd;

  if(XAllocNamedColor(dpy, DefaultColormap(dpy, DefaultScreen(dpy)),
		      rd_str, &color, &unused) == 0)
    {
      throw BadColorName("graphic1");
    }
  gcv.foreground = color.pixel;
  gc_rd = XCreateGC(dpy, win,
		    GCForeground,
		    &gcv);
  
  if(wr_str == NULL)
    wr_str = default_wr;

  if(XAllocNamedColor(dpy, DefaultColormap(dpy, DefaultScreen(dpy)),
		      wr_str, &color, &unused) == 0)
    {
      throw BadColorName("graphic2");
    }
  gcv.foreground = color.pixel;
  gc_wr = XCreateGC(dpy, win,
		    GCForeground,
		    &gcv);
}

void IOAct1::get_val(unsigned long *rio, unsigned long *wio) {
  int fd;
  char buff[255];
  char ch;
  unsigned int i;

  fd = open("/proc/stat", O_RDONLY);
  if(fd < 0) {
    perror("open");
    exit(0);
  }

  while(read(fd, &ch, 1) > 0) {
    if(ch == '\n' && read(fd, buff, 8) == 8) {
      if(strncmp(buff, "disk_rio", 8) == 0) {
	i = 0;
	while(read(fd, &ch, 1) > 0 && ((ch >= '0' && ch <= '9') || ch == ' '))
	  buff[i++] = ch;
	buff[i] = '\0';
	sscanf(buff, "%lu %*u %*u %*u", rio);

	read(fd, buff, 8);
	if(strncmp(buff, "disk_wio", 8) == 0) {
	  i = 0;
	  while(read(fd, &ch, 1) > 0 && ((ch >= '0' && ch <= '9') || ch == ' '))
	    buff[i++] = ch;
	  buff[i] = '\0';
	  sscanf(buff, "%lu %*u %*u %*u", wio);

	  goto end;
	}
      }
    }
  }

 end:
  close(fd);
  return;
}

void IOAct1::draw() {
  char           strval[20];
  unsigned int   i, n;
  unsigned int   size, offset;
  unsigned int   half_asc, half;
  unsigned int   top, bottom;
  unsigned int   start, stop;
  unsigned int   val;
  int            asc, desc, dir;
  XCharStruct    cst;

  /* efface le fond */
  XFillRectangle(dpy, pix, gc_bg, 0, 0, width, height);

  /* dessine l'chelle sur le cot et la borne maximale */
  n = sprintf(strval, "%llu", max);
  XTextExtents(fst, " ", 3, &dir, &asc, &desc, &cst);
  XDrawString(dpy, pix, gc_fg,
	      2, 2 + asc,
	      strval, n);
  offset = 2 + XTextWidth(fst, strval, n) + 2;

  half_asc = asc >> 1;

  /* ticks */
  top = 2 + half_asc;
  bottom = height - 1;
  XDrawLine(dpy, pix, gc_fg, offset, top, offset + 3, top);
  XDrawLine(dpy, pix, gc_fg, offset, bottom, offset + 3, bottom);
  half = top + (bottom - top) / 2;
  XDrawLine(dpy, pix, gc_fg, offset, half, offset + 3, half);

  /* vertical scale */
  XDrawLine(dpy, pix, gc_fg, offset + 3, top, offset + 3, bottom+1);
  offset += 5;

  size = cbuff_real_size(&io);

  n = width - offset;
  start = (size < n) ? 0 : size-n;
  stop = size - start;

  /* WARNING : a division by zero may occur */
  for(i = 0; i < stop; i++) {
    if((val = cbuff_get(&io, start+i)))
      XDrawLine(dpy, pix, gc_wr,
		i + offset, bottom,
		i + offset, bottom - (int)((double)val/(double)max*(double)(bottom - top)));

    if((val = cbuff_get(&rio, start+i)))
      XDrawLine(dpy, pix, gc_rd,
		i + offset, bottom,
		i + offset, bottom - (int)((double)val/(double)max*(double)(bottom - top)));
  }

  /* Draw infos */
  XDrawString(dpy, pix, gc_fg,
	      width - XTextWidth(fst, "R/W IO", 6) - 1,
	      2 + asc,
	      "R/W IO", 6);
}

void IOAct1::update() {
  unsigned long  rdiff, tdiff;
  unsigned int   old_last_r, old_last_t;
  unsigned long  rio_val, wio_val;

  get_val(&rio_val, &wio_val);
  rdiff = rio_val - lastread;
  tdiff = rio_val + wio_val - lasttot;

  old_last_r = cbuff_get(&rio, 0);
  cbuff_add(&rio, rdiff);

  old_last_t = cbuff_get(&io, 0);
  cbuff_add(&io, tdiff);

  if(cbuff_real_size(&io) == io.window && old_last_t == max/* && diff < irq_st.max*/) {
    unsigned int i, size, val;

    size = cbuff_real_size(&io);
    max = 0;
    for(i = 0; i < size; i++) {
      val = cbuff_get(&io, i);
      max = (val > max) ? val : max;
    }
  }
  else
    max = (max < tdiff) ? tdiff : max;

  lastread = rio_val;
  lasttot = rio_val + wio_val;

  draw();
}

void IOAct1::resize(unsigned int w, unsigned int h) {
  PixBuff::resize(w, h);
  cbuff_resize(&io, w);
  cbuff_resize(&rio, w);
}
